/* SetPriX - SysSetPriority/DosSetPriority wrapper

   Copyright (c) 2024-2025 Steven Levine and Associates, Inc.
   All rights reserved.

   2024-10-13 SHL Baseline
   2024-11-01 SHL Correct alpha class calc
   2024-11-02 SHL Show priority etc if verbose or no args
   2025-12-12 SHL More usage
*/

signal on Error
signal on Failure name Error
signal on Halt
signal on NotReady name Error
signal on NoValue name Error
signal on Syntax name Error

gVersion = '0.1 2025-12-12'

Globals = 'gClass gCmdName gDelta gEnv gVerbose gVersion'

call Initialize

Main:

  parse arg cmdLine
  call ScanArgs cmdLine
  drop cmdLine

  if gClass \== '' then do
    if gVerbose then
      say 'Setting priority class to' gClass 'and delta to' gDelta
    call SysSetPriority gClass, gDelta
    if RESULT \= 0 then
      call Die 'SysSetPriority reported error' RESULT
  end

  if gClass == '' | gVerbose then
    call ShowProcessInfo

  exit

/* end Main */

/*=== ShowProcessInfo() Do one thing ===*/

ShowProcessInfo: procedure expose (Globals)
  pid = 0
  info = Sys2QueryProcess(pid, 'P')

  say
  say 'pid         ppid ptype pri  cpu time'
  parse var info p pp pt pri cputime tail
  say left(p, 4) left('('d2x(p)')', 6) left(pp, 4) left(pt, 5) pri cputime
  return

/* end ShowProcessInfo */

/*=== Initialize() Initialize globals ===*/

Initialize: procedure expose (Globals)
  call SetCmdName
  call LoadRexxUtil
  call LoadRxUtilEx
  gEnv = 'OS2ENVIRONMENT'
  return

/* end Initialize */

/*=== ScanArgsInit() ScanArgs initialization exit routine ===*/

ScanArgsInit: procedure expose (Globals) cmdTail swCtl keepQuoted

  /* Preset defaults */
  gClass = ''
  gDelta = ''
  gVerbose = 0				/* Verbose mode level */

  return

/* end ScanArgsInit */

/*=== ScanArgsSwitch() ScanArgs switch option exit routine ===*/

ScanArgsSwitch: procedure expose (Globals) curSw curSwArg

  select
  when curSw == 'h' | curSw == '?' then
    call ScanArgsHelp
  when curSw == 'v' then
    gVerbose = gVerbose + 1
  when curSw == 'V' then do
    say
    say gCmdName gVersion
    exit
  end
  otherwise
    call ScanArgsUsage 'Switch "-' || curSw || '" unexpected'
  end /* select */

  return

/* end ScanArgsSwitch */

/*=== ScanArgsArg() ScanArgs argument option exit routine ===*/

ScanArgsArg: procedure expose (Globals) curArg

  if gClass == '' then do
    if curArg = '.' then
      curArg = 0			/* Map to no change */
    if datatype(curArg, 'W') then do
      if curArg < 0 | curArg > 4 then
	call ScanArgsUsage 'Class must be between 0 and 4'
      gClass = curArg
    end
    else do
      className = left(translate(curArg), 1)
      class = pos(className, 'NIRTF')
      if class = 0 then
	call ScanArgsUsage 'Class names are N I R T or F'
      gClass = class - 1
    end
  end
  else if gDelta == '' then do
    if \ datatype(curArg, 'W') | curArg < -31 | curArg > 31 then
      call ScanArgsUsage 'Delta must be between -31 and 31'
    else
      gDelta = curArg
  end
  else
    call ScanArgsUsage 'Class and delta already specified'

  return

/* end ScanArgsArg */

/*=== ScanArgsTerm() ScanArgs scan end exit routine ===*/

ScanArgsTerm: procedure expose (Globals)

  if gClass \== '' & gDelta == '' then
    gDelta = 0				/* Default to 0 */
  return

/* end ScanArgsTerm */

/*=== ScanArgsHelp() Display ScanArgs usage help exit routine ===*/

ScanArgsHelp:
  say
  say 'SysSetPriority/DosSetPriority wrapper'
  say
  say 'Class is class number or first letter of class name'
  say 'Priority range is 0 to 31'
  say 'Delta is priority delta'
  say 'Attempts to set priority less than 0 are silently ignored'
  say 'Run with no arguments to display current settings'
  say
  say 'Usage:' gCmdName '[-h] [-v] [-V] [-?] [class [delta]]'
  say
  say '  -h -?  Display this message'
  say '  -v     Enable verbose messages, repeat for more verbosity'
  say '  -V     Display version number and quit'
  say
  say '  class  0-NoChange 1-Idle 2-Regular  4=Foreground 3-Time Critical'
  say '         class is class number or first letter of class name'
  say '  delta  -31 to +31, default is 0'
  exit 255

/* end ScanArgsHelp */

/*=== ScanArgsUsage(message) Report ScanArgs usage error exit routine ===*/

ScanArgsUsage:
  parse arg msg
  say
  if msg \== '' then
    say msg
  say 'Usage:' gCmdName '[-h] [-v] [-V] [-?] [class [delta]]'
  exit 255

/* end ScanArgsUsage */

/*==============================================================================*/
/*=== SkelRexxFunc standards - Delete unused - Move modified above this mark ===*/
/*==============================================================================*/

/*==========================================================================*/
/*=== SkelRexx standards - Delete unused - Move modified above this mark ===*/
/*==========================================================================*/

/*=== Die([message,...]) Write multi-line message to STDERR and die ===*/

Die:
  call lineout 'STDERR', ''
  do i = 1 to arg()
    call lineout 'STDERR', arg(i)
  end
  call lineout 'STDERR', gCmdName 'aborting at line' SIGL || '.'
  call beep 200, 300
  call SysSleep 2
  exit 254

/* end Die */

/*=== Error() Set gErrCondition; report to STDOUT; trace and exit or return if called ===*/

Error:
  say
  parse source . . cmd
  gErrCondition = condition('C')
  say gErrCondition 'signaled at line' SIGL 'of' cmd || '.'
  if condition('D') \== '' then
    say 'REXX reason =' condition('D') || '.'
  if gErrCondition == 'SYNTAX' & symbol('RC') == 'VAR' then
    say 'REXX error =' RC '-' errortext(RC) || '.'
  else if symbol('RC') == 'VAR' then
    say 'RC =' RC || '.'
  say 'Source =' sourceline(SIGL)

  if condition('I') \== 'CALL' | gErrCondition == 'NOVALUE' | gErrCondition == 'SYNTAX' then do
    trace '?A'
    say 'Enter REXX commands to debug failure.  Press enter to exit script.'
    nop
    if symbol('RC') \== 'VAR' then
      RC = 255
    exit RC
  end

  return

/* end Error */

/*=== Halt() Report HALT condition to STDOUT and exit ===*/

Halt:
  say
  parse source . . cmd
  say condition('C') 'signaled at' cmd 'line' SIGL || '.'
  say 'Source =' sourceline(SIGL)
  say 'Sleeping for 2 seconds...'
  call SysSleep 2
  exit 253

/* end Halt */

/*=== LoadRexxUtil() Load RexxUtil functions ===*/

LoadRexxUtil:
  if RxFuncQuery('SysLoadFuncs') then do
    call RxFuncAdd 'SysLoadFuncs', 'REXXUTIL', 'SysLoadFuncs'
    if RESULT then
      call Die 'Cannot load SysLoadFuncs.'
    call SysLoadFuncs
  end
  return

/* end LoadRexxUtil */

/*=== LoadRxUtilEx() Load Alex's RxUtilEx functions ===*/

LoadRxUtilEx:
  if RxFuncQuery('Sys2LoadFuncs') then do
    call RxFuncAdd 'Sys2LoadFuncs', 'RXUTILEX', 'Sys2LoadFuncs'
    if RESULT then
      call Die 'Cannot load Sys2LoadFuncs'
    call Sys2LoadFuncs
  end
  return

/* end LoadRxUtilEx */

/*=== ScanArgs(cmdLine) Scan command line ===*/

ScanArgs: procedure expose (Globals)

  /* Calls user exits to process arguments and switches */

  parse arg cmdTail
  cmdTail = strip(cmdTail)

  call ScanArgsInit

  /* Ensure optional settings initialized */
  if symbol('SWCTL') \== 'VAR' then
    swCtl = ''				/* Switches that take args, append ? if optional */
  if symbol('KEEPQUOTED') \== 'VAR' then
    keepQuoted = 0			/* Set to 1 to keep arguments quoted */

  /* Scan */
  curArg = ''				/* Current arg string */
  curSwList = ''			/* Current switch list */
  /* curSwArg = '' */			/* Current switch argument, if needed */
  noMoreSw = 0				/* End of switches */

  do while cmdTail \== '' | curArg \== '' | curSwList \== ''

    /* If arg buffer empty, refill */
    if curArg == '' then do
      qChar = left(cmdTail, 1)		/* Remember quote */
      if \ verify(qChar,'''"', 'M') then
	parse var cmdTail curArg cmdTail	/* Not quoted */
      else do
	/* Arg is quoted */
	curArg = ''
	do forever
	  /* Parse dropping quotes */
	  parse var cmdTail (qChar)quotedPart(qChar) cmdTail
	  curArg = curArg || quotedPart
	  /* Check for escaped quote within quoted string (i.e. "" or '') */
	  if left(cmdTail, 1) \== qChar then do
	    cmdTail = strip(cmdTail)	/* Strip leading whitespace */
	    leave			/* Done with this quoted arg */
	  end
	  curArg = curArg || qChar	/* Append quote */
	  if keepQuoted then
	    curArg = curArg || qChar	/* Append escaped quote */
	  parse var cmdTail (qChar) cmdTail	/* Strip quote */
	end /* do forever */
	if keepQuoted then
	  curArg = qChar || curArg || qChar	/* requote */
      end /* if quoted */
    end /* if curArg empty */

    /* If switch buffer empty, refill */
    if curSwList == '' & \ noMoreSw then do
      if left(curArg, 1) == '-' & curArg \== '-' then do
	if curArg == '--' then
	  noMoreSw = 1
	else
	  curSwList = substr(curArg, 2)	/* Remember switch string */
	curArg = ''			/* Mark empty */
	iterate				/* Refill arg buffer */
      end /* if switch */
    end /* if curSwList empty */

    /* If switch in progress */
    if curSwList \== '' then do
      curSw = left(curSwList, 1)	/* Next switch */
      curSwList = substr(curSwList, 2)	/* Drop from pending */
      /* Check switch allows argument, avoid matching ? */
      if pos(curSw, translate(swCtl,,'?')) \= 0 then do
	if curSwList \== '' then do
	  curSwArg = curSwList		/* Use rest of switch string for switch argument */
	  curSwList = ''
	end
	else if curArg \== '' & left(curArg, 1) \== '-' then do
	  curSwArg = curArg		/* Arg string is switch argument */
	  curArg = ''			/* Mark arg string empty */
	end
	else if pos(curSw'?', swCtl) = 0 then
	  call ScanArgsUsage 'Switch "-' || curSw || '" requires an argument'
	else
	  curSwArg = ''			/* Optional arg omitted */
      end

      call ScanArgsSwitch		/* Passing curSw and curSwArg */
      drop curSwArg			/* Must be used by now */
    end /* if switch */

    /* If arg */
    else if curArg \== '' then do
      noMoreSw = 1
      call ScanArgsArg			/* Passing curArg */
      curArg = ''
    end

  end /* while not done */

  call ScanArgsTerm

  return

/* end ScanArgs */

/*=== SetCmdName() Set gCmdName to short script name ===*/

SetCmdName: procedure expose (Globals)
  parse source . . cmd
  cmd = filespec('N', cmd)		/* Chop path */
  c = lastpos('.', cmd)
  if c > 1 then
    cmd = left(cmd, c - 1)		/* Chop extension */
  gCmdName = translate(cmd, xrange('a', 'z'), xrange('A', 'Z'))	/* Lowercase */
  return

/* end SetCmdName */

/* eof */
