/* ChkCfg - Check CONFIG.SYS semantics or environment

   Copyright (c) 1997-2022 Steven Levine and Associates, Inc.
   All rights reserved.

   1997-09-07 SHL Baseline
   2017-08-31 SHL Support set statement with optional trailing semicolon
   2017-08-31 SHL Convert to Globals style
   2018-10-26 SHL Support more statements; be more generic
   2018-11-07 SHL Warn if LIBPATH directory empty
   2018-11-08 SHL Tweak messages
   2019-05-26 SHL Tweak help
   2020-04-28 SHL Sync with templates
   2020-04-28 SHL Support HOMEDRIVE HOMEPATH combo
   2020-06-22 SHL Warn if directory empty
   2020-06-26 SHL Warn if directory defined more than once
   2020-06-26 SHL Detect more spurious whitespace
   2020-06-26 SHL Detect spurious ;;
   2021-12-03 SHL Sync with templates
   2021-12-03 SHL Drop unused code
   2021-12-03 SHL Correct missing ||
   2022-03-23 SHL Sync with templates
   2022-03-23 SHL If -e check relative to current current drive if drive not specified
   2022-08-09 SHL Correct typo
*/

signal on Error
signal on Failure name Error
signal on Halt
signal on NotReady name Error
signal on NoValue name Error
signal on Syntax name Error

gVersion = '0.6 2022-03-23'

Globals = 'gArgList. gBootDrive gChkEnv gCmdName gCurDrv gDbgLvl',
	  'gEnv gErrCondition gErrLineNum gHomeDrv',
	  'gLine gLineNum gTesting gVerbose gVersion'

call Initialize

Main:

  parse arg cmdLine
  call ScanArgs cmdLine
  drop cmdLine

  gErrLineNum = 0

  if gChkEnv then do
    gBootDrive = SysBootDrive()
    gCurDrv = left(directory(), 2)	/* Only used if gChkEnv */
    call ChkEnv
  end
  else do
    do argNum = 1 to gArgList.0
      inputFile = gArgList.argNum
      call DoOneFile inputFile
    end
  end

  exit

/* end main */

/*=== ChkEnv() Check environment ===*/

ChkEnv: procedure expose (Globals)

  '@set | rxqueue'

  env.0 = 0
  do while queued() > 0
    pull s
    i = env.0 + 1
    env.i = s
    env.0 = i
  end

  do i = 1 to env.0
    s = 'SET' env.i
    gLineNum = i
    gLine = s
    call ChkOneLine
  end
  return

/* end ChkEnv */

/*=== DoOneFile(fileName) Do one config.sys file ===*/

DoOneFile: procedure expose (Globals)

  parse arg inputFile
  call WarnMsg 'Checking' inputFile

  /* Scan and parse */

  gBootDrive = SysBootDrive()

  if \ IsFile(inputFile) then
    call ScanArgsUsage 'Cannot access' inputFile 'file'

  gBootDrive = filespec('D', stream(inputFile, 'C', 'QUERY EXISTS'))
  if gBootDrive = '' then
    call Die 'Cannot determine drive for' inputFile

  call stream inputFile, 'C', 'OPEN READ'

  drop ErrCondition

  lines.0 = 0

  do while lines(inputFile) \= 0
    call on NOTREADY name CatchError	/* Avoid death on missing NL */
    line = linein(inputFile)
    signal on NOTREADY name Error
    lineNum = lines.0 + 1
    lines.lineNum = line
    lines.0 = lineNum
  end /* while lines */

  call stream inputFile, 'C', 'CLOSE'

  do lineNum = 1 to lines.0
    gLineNum = lineNum
    gLine = lines.lineNum
    call ChkOneLine
  end

  call WarnMsg 'Processed' lines.0 'lines'
  return

/* end DoOneFile */

/*=== ChkBaseDev(val) Check BASEDEV ===*/

ChkBaseDev: procedure expose (Globals)

  parse arg tail

  fileName = tail

  /* Strip args */
  i = FindWhite(fileName)
  if i > 0 then
    fileName = left(fileName, i - 1)

  do 1
    if length(fileName) = 0 then do
      call ErrMsg '"' || tail || '" does not contain a file name'
      leave
    end

    s = gBootDrive || '\OS2\BOOT\' || fileName
    if IsFile(s) then
      leave

    s = gBootDrive || '\OS2\' || fileName
    if IsFile(s) then
      leave

    s = gBootDrive || '\' || fileName
    if IsFile(s) then
      leave

    call ErrMsg 'Cannot access' '"' || fileName || '" as a basedev file name'

  end /* do */

  return

/* end ChkBaseDev */

/*=== ChkDir(directory) Check single directory ===*/

ChkDir: procedure expose (Globals)

  parse arg path

  s = ChopPctN(path)

  if \ IsDrvRdy(s) then
    call ErrMsg 'Cannot access drive for "' || path || '"'
  else if \ IsDir(s) then
    call ErrMsg 'Cannot access "' || path || '" as a directory'
  else if \ IsFile(s || '\*') then
    call ErrMsg '"' || path || '" is empty'

  return

/* end ChkDir */

/*=== ChkDirOrFileList(list) Check list of directories or file names ===*/

ChkDirOrFileList: procedure expose (Globals)

  parse arg tail

  do while length(tail) \= 0

    /* Extract next list item */
    i = pos(';', tail)
    if i = 0 then do
      fileName = tail
      tail = ''
    end
    else do
      fileName = substr(tail, 1, i - 1)
      tail = substr(tail, i + 1)
    end

    if fileName == '' then do
      call ErrMsg 'Statement contains spurious ;;'
      iterate
    end

    if left(fileName, 1) == ' ' then
      call ErrMsg '"' || fileName || '" contains unexpected whitespace'
    if right(fileName, 1) == ' ' then
      call ErrMsg '"' || fileName || '" contains unexpected whitespace'

    call DbgMsg 2, ' Checking' fileName
    if \ IsDrvRdy(fileName) then
      call ErrMsg 'Cannot access drive for "' || fileName || '"'
    else if \ IsFile(fileName) & \ IsDir(fileName) then
      call ErrMsg 'Cannot access "' || fileName || '" as a directory or file'

  end /* while */

  return

/* end ChkDirOrFileList */

/*=== ChkDirList(list, chkempty) Check directory name list ===*/

ChkDirList: procedure expose (Globals)

  parse arg list, chkempty

  chkempty = chkempty \== '' & chkempty \= 0

  dirList.0 = 0

  /* List is semicolon separated with optional trailing semicolon
     chkempty warns if directory exists and is empty - used for LIBPATH
     Leading spaces are significant
     Trailing spaces ignored but warn anyway
  */

  do while length(list) \= 0

    /* Extract next item from list */
    i = pos(';', list)
    if i = 0 then do
      rawPath = list
      list = ''
    end
    else do
      rawPath = substr(list, 1, i - 1)
      list = substr(list, i + 1)
    end

    if rawPath == '' then do
      call ErrMsg 'Statement contains spurious ;;'
      iterate
    end

    if left(rawPath, 1) == ' ' then
      call ErrMsg '"' || rawPath || '" contains unexpected whitespace'
    if right(rawPath, 1) == ' ' then
      call ErrMsg '"' || rawPath || '" contains unexpected whitespace'

    wrkPath = ChopPctN(rawPath)

    wrkPath = strip(wrkPath)		/* Assume whitespace spurious */

    /* Supply boot drive if omitted */
    if wrkPath \== '.' then do
      drv = filespec('D', wrkPath)
      if drv == '' then do
	if gChkEnv then
	  drv = gCurDrv
	else
	  drv = gBootDrive

	wrkPath = drv || wrkPath
      end
    end

    /* Check for dup occurences */
    s = translate(wrkPath)
    if symbol('dirList.s') \== 'VAR' then
      dirList.s = wrkPath
    else do
      t = dirList.s
      call ErrMsg '"' || rawPath || '" already defined as "' || t || '"'
    end

    /* Check access */
    if \ IsDrvRdy(wrkPath) then
      call ErrMsg 'Cannot access drive for "' || rawPath || '"'
    else if \ IsDir(wrkPath) then
      call ErrMsg 'Cannot access "' || rawPath || '" as a directory'
    else if chkempty then do
      /* Check directory empty */
      wildcard = wrkPath
      if right(wildcard, 1) \== '\' then
	wildcard = wildcard || '\'
      wildcard = wildcard || '*'
      call SysFileTree wildcard, 'fileList'
      if fileList.0 = 0 then
	call ErrMsg '"' || rawPath || '" is an empty LIBPATH directory'
    end

  end /* while */

  return

/* end ChkDirList */

/*=== ChkFileList(list) Check file name list ===*/

ChkFileList: procedure expose (Globals)

  parse arg list

  /* List is semicolon separated with optional trailing semicolon */

  do while length(list) \= 0

    /* Extract next list item */
    i = pos(';', list)
    if i = 0 then do
      fileName = list
      list = ''
    end
    else do
      fileName = substr(list, 1, i - 1)
      list = substr(list, i + 1)
    end

    if fileName == '' then do
      call ErrMsg 'Statement contains spurious ;;'
      iterate
    end

    if left(fileName, 1) == ' ' then
      call ErrMsg '"' || fileName || '" contains unexpected whitespace'
    if right(fileName, 1) == ' ' then
      call ErrMsg '"' || fileName || '" contains unexpected whitespace'

    /* Ignore if no directory separators */
    if pos('\', fileName) = 0 then
      iterate

    call DbgMsg 2, 'Checking' fileName

    if IsFile(fileName) then
      iterate

    if left(fileName, 1) == '\' then do
      if gChkEnv then
        drv = gCurDrv
      else
        drv = gBootDrive
      s = drv || fileName
      if IsFile(s) then
	leave
    end

    call ErrMsg 'Cannot access "' || fileName || '" as a file'

  end /* while */

  return

/* end ChkFileList */

/*=== ChkOneFile(val, hasArgs) Check one file name ===*/

ChkOneFile: procedure expose (Globals)

  parse arg tail, hasArgs

  hasArgs = hasArgs \== '' & hasArgs \= 0

  /* Strip white around = */
  if left(tail, 1) == '=' then do
    tail = substr(tail, 2)
    i = FindNonWhite(tail)
    if i > 1 then
      tail = substr(tail, i)
  end

  /* tail is a file name and depending on caller may have args */
  fileName = tail

  if hasArgs then do
    i = FindWhite(fileName)
    if i > 0 then
      fileName = left(fileName, i - 1)
  end

  do 1
    if length(fileName) = 0 then do
      call ErrMsg '"' || tail || '" does not contain a file name'
      leave
    end

    if IsFile(fileName) then
      leave

    if left(fileName, 1) == '\' then do
      s = gBootDrive || fileName
      if IsFile(s) then
	leave
    end

    call ErrMsg 'Cannot access "' || fileName || '" as a file'

  end /* do */

  return

/* end ChkOneFile */

/*=== ChkOneLine() check one line ===*/

ChkOneLine: procedure expose (Globals)

  iStart = FindNonWhite(gLine)		/* Skip leading whitespace */

  if iStart = 0 then
    return

  tail = substr(gLine, iStart)

  iEnd = FindDelim(tail)		/* Find delimiter */

  if iEnd = 0 then do
    cmd = tail				/* No delimiter */
    tail = ''
  end
  else do
    cmd = substr(tail, 1, iEnd - 1)
    delim = substr(tail, iEnd, 1)
    tail = substr(tail, iEnd + 1)	/* Tail includes any leading whitespace */
  end

  cmd = translate(cmd)

  /* See ErrMsg */
  if gVerbose & cmd \== 'REM' then do
    gErrLineNum = gLineNum		/* Avoid dups */
    s = 'Checking' gLineNum':' gLine
    if length(s) > 76 then
      s = left(s, 76) || '...'
    say s
  end

  call DbgMsg 3, 'CMD '''cmd''''

  cmdWithFileAndArgs = ';CALL' ||,
		       ';DEVICE' ||,
		       ';IFS' ||,
		       ';RUN' ||,
		       ';SHELL' ||,
		       ';'

  cmdWithFile = ';PROTSHELL' ||,
		';'

  select
  when cmd == 'REM' then
    nop
  when pos(';'cmd';', cmdWithFileAndArgs) > 0 then do
    call DbgMsg 'Checking' cmd
    call ChkOneFile tail, 1
  end
  when pos(';'cmd';', cmdWithFile) > 0 then do
    call DbgMsg 'Checking' cmd
    call ChkOneFile tail
  end
  when cmd == 'SET' then do
    call ChkSetStmt tail
  end /* set */
  when cmd == 'BASEDEV' then do
    call DbgMsg 'Checking BASEDEV'
    call ChkBaseDev tail
  end
  when cmd == 'LIBPATH' then do
    call DbgMsg 'Checking LIBPATH'
    call ChkDirList tail, 1
    nop
  end
  otherwise do
    call DbgMsg 'Skipped' gLine
    nop
  end /* otherwise */
  end /* select */

  return

/* end ChkOneLine */

/*=== ChkSetStmt(name=value) check set statement ===*/

ChkSetStmt: procedure expose (Globals)

  parse arg tail

  /* Set statements that are ignored for now */
  varIsSkipped = ';LIBC_HOOK_DLLS' ||,
			  ';'

  /* Set statements containing single directory name and no semicolon */
  valIsOneDir = ';BA2_CATALOG_PATH' ||,
		';BA2_LOG_PATH' ||,
		';BA2_SET_PATH' ||,
		';CPE' ||,
		';CPPHELP_INI' ||,
		';CPPLOCAL' ||,
		';CPPMAIN' ||,
		';CPPWORK' ||,
		';DMIPATH' ||,
		';DSSDIR' ||,
		';DSSPATH' ||,
		';ETC' ||,
		';GNUPLOT' ||,
		';GS_LIB' ||,
		';HOME' ||,
		';I18NDIR' ||,
		';IBMAV' ||,
		';INIT' ||,
		';IWFOPT' ||,
		';LANINSTEP' ||,
		';LOTUS_CLS' ||,
		';NETVIEW_PATH' ||,
		';REFER' ||,
		';OCRNOTES' ||,
		';OCTAVE_HOME' ||,
		';PROTODIR' ||,
		';SMTMP' ||,
		';SNMPDIR' ||,
		';SVA_PATH' ||,
		';TEMP' ||,
		';TMP' ||,
		';TMPDIR' ||,
		';TMPS' ||,
		';'

  /* Set statements containing single directory name and optional semicolon */
  valIsOneDirWithOptSemi = ';DSPPATH' ||,
			   ';'

  /* Set statements that define a list of files */
  valIsFileList = ';SOMIR' ||,
		  ';'

  /* Set statements that define a list of files */
  valIsOneFile = ';BFILE' ||,
		 ';COMSPEC' ||,
		 ';DLSINI' ||,
		 ';HTML_TIDY' ||,
		 ';OS2_SHELL' ||,
		 ';RUNWORKPLACE' ||,
		 ';SCFINDUTILITY' ||,
		 ';SYSTEM_INI' ||,
		 ';TERMCAP' ||,
		 ';TLIBCFG' ||,
		 ';USER_INI' ||,
		 ';'

  /* Set statements that define a list of directories and files */
  valIsDirAndFileList = ';CLASSPATH' ||,
			  ';'

  /* Others ending in PATH assumed to define a set of directories */
  /* Others ending in FILE assumed to define a set of files */
  /* Others that look like a path list are assume to define a set of directories */
  /* Anything else is ignored */

  iEnd = FindDelim(tail)		/* Find = or whitespace */
  if iEnd = 0 then do
    call ErrMsg 'Cannot find expected whitespace or = delimiter in "' || tail || '"'
    exit
  end
  var = substr(tail, 1, iEnd - 1)
  call DbgMsg 3, '-> var:' ''''var''''
  tail = substr(tail, iEnd)
  iEnd = FindNonWhite(tail)
  if iEnd = 0 then do
    call ErrMsg 'Cannot find expected whitespace or = delimiter in "' || tail || '"'
    exit
  end
  delim = substr(tail, iEnd, 1)
  if delim \== '=' then do
    call ErrMsg 'Cannot find expected = delimiter in "' || tail || '"'
    exit
  end
  tail = substr(tail, iEnd + 1)		/* Value includes whitespace */
  iEnd = FindNonWhite(tail)
  if iEnd = 0 then do
    call ErrMsg 'Cannot find delimiter in "' || tail || '"'
    exit
  end
  val = substr(tail, iEnd)
  /* Check special cases first
     Otherwise use variable name to guess format
     Otherwise used value layout to guess
     HOMEPATH is special case
     Otherwise ignore
  */
  select
  when pos(';'var';', varIsSkipped) \= 0 then do
    call DbgMsg 'Skipped' gLine
  end
  when pos(';'var';', valIsOneDir) \= 0 then do
    call DbgMsg 'Checking' var
    call ChkDir val
  end
  when pos(';'var';', valIsOneDirWithOptSemi) \= 0 then do
    call DbgMsg 'Checking' var
    val = strip(val, 'T', ';')
    call ChkDir val
  end
  when pos(';'var';', valIsFileList) \= 0 then do
    call DbgMsg 'Checking' var
    call ChkFileList val
  end
  when pos(';'var';', valIsOneFile) \= 0 then do
    call DbgMsg 'Checking' var
    call ChkOneFile val
  end
  when pos(';'var';', valIsDirAndFileList) \= 0 then do
    call DbgMsg 'Checking' var
    call ChkDirOrFileList val
  end
  when var == 'HOMEPATH' then do
    if symbol('gHomeDrv') == 'VAR' then
      call ChkDirList gHomeDrv || val	/* Assume HOMEPATH sane */
    else do
      call WarnMsg 'HOMEDRIVE must be set before HOMEPATH'
      call ChkDirList val
    end
  end
  when translate(right(var, 4)) == 'PATH' then do
    call DbgMsg 'Checking' var
    call ChkDirList val
  end
  when translate(right(var, 4)) == 'FILE' then do
    call DbgMsg 'Checking' var
    call ChkFileList val
  end
  when var == 'HOMEDRIVE' then
    gHomeDrv = val
  /* If looks likes path list */
  when (pos(translate(left(val, 1)), xrange('C', 'Z')) > 0 &,
       substr(val, 2, 2) == ':\' ) |,
       left(val, 2) == '.;' |,
       val == '.'
  then do
    call DbgMsg 'Checking' var
    call ChkDirList val
  end
  otherwise do
    call DbgMsg 'Skipped' gLine
  end /* otherwise */
  end /* select */

  return

/* end ChkSetStmt */

/*=== ChopPctN(directory) Replace %N with LANG value ===*/

ChopPctN: procedure expose (Globals)
  parse arg dir
  if right(dir, 3) == '\%N' then
    dir = substr(dir, 1, length(dir) - 3)
  else do
    i = pos('\%L\', dir)
    if i > 0 then do
      lang = value('LANG',, gEnv)
      if lang \== '' then
	dir = substr(dir, 1, i) || lang || substr(dir, i + 3)
    end
  end
  return dir

/* end ChopDirSlash */

/*=== DbgMsg([level, ]message) Display multi-line message to STDOUT if debugging ===*/

DbgMsg: procedure expose (Globals)

  /* Like standard DbgMsg, but output to stdout, not stderr */
  dbgLvl = arg(1)
  if dataType(dbgLvl, 'W') then
    start = 2
  else do
    dbgLvl = 1
    start = 1
  end
  if dbgLvl <= gDbgLvl then do
    do i = start to arg()
      msg = arg(i)
      if msg \== '' then
	msg = ' *' msg
      say msg
    end
  end
  return

/* end DbgMsg */

/*=== ErrMsg(message,...) Display multi-line message to STDERR ===*/

ErrMsg: procedure expose (Globals)

  if gLineNum \= gErrLineNum then do
    gErrLineNum = gLineNum		/* Avoid dups */
    s = gLineNum':' gLine
    if length(s) > 76 then
      s = left(s, 76) || '...'
    say s
  end

  do i = 1 to arg()
    msg = arg(i)
    if msg \== '' then
      msg = ' *' msg
    say msg
  end
  return

/* end ErrMsg */

/*=== FindDelim(line) Find first blank, tab or = and return index ===*/

FindDelim: procedure

  parse arg s

  iBlank = pos(' ', s)
  iTab = pos(x2c(9), s)
  iEq = pos('=', s)

  iDelim = max(iBlank, iTab, iEq)

  if iBlank > 0 then
    iDelim = min(iDelim, iBlank)
  if iTab > 0 then
    iDelim = min(iDelim, iTab)
  if iEq > 0 then
    iDelim = min(iDelim, iEq)

  return iDelim

/* end FindDelim */

/*=== FindWhite(line) Find blank or tab and return index ===*/

FindWhite: procedure

  parse arg s

  iBlank = pos(' ', s)
  iTab = pos(x2c(9), s)

  iWhite = max(iBlank, iTab)

  if iBlank > 0 then
    iWhite = min(iWhite, iBlank)
  if iTab > 0 then
    iWhite = min(iWhite, iTab)

  return iWhite

/* end FindWhite */

/*=== FindNonWhite(line) Findnext Non blank or tab and return index or 0 ===*/

FindNonWhite: procedure

  parse arg s

  do i = 1 to length(s)
    ch = substr(s, i, 1)
    if ch \= ' ' & ch \= x2c(9) then
      leave
  end

  if i > length(s) then
    i = 0

  return i

/* end FindNonWhite */

/*=== VerboseMsg(message,...) Display multi line message to STDOUT if verbose ===*/

VerboseMsg: procedure expose (Globals)
  /* Like standard VerboseMsg, but output to STDOUT not STDERR */
  if gVerbose then do
    do i = 1 to arg()
      say arg(i)
    end
  end
  return

/* end VerboseMsg */

/*=== WarnMsg(message,...) Write multi-line warning message to STDOUT ===*/

WarnMsg: procedure
  /* Like standard WarnMsg, but output to STDOUT not STDERR */
  do i = 1 to arg()
    say arg(i)
  end
  return

/* end WarnMsg */

/*=== Initialize() Initialize globals ===*/

Initialize:

  call SetCmdName
  call LoadRexxUtil
  gEnv = 'OS2ENVIRONMENT'
  return

/* end Initialize */

/*=== ScanArgsInit() ScanArgs initialization exit routine ===*/

ScanArgsInit: procedure expose (Globals) cmdTail swCtl keepQuoted

  /* Preset defaults */
  gDbgLvl = 0				/* Display debug messages */
  gChkEnv = 0				/* Check environment */
  gVerbose = 0				/* Verbose messages */
  gArgList.0 = 0			/* Reset arg count */

  /* Configure scanner */
  swCtl = ''				/* Switches that take args, append ? if arg optional */
  keepQuoted = 0			/* Set to 1 to keep arguments quoted */

  return

/* end ScanArgsInit */

/*=== ScanArgsSwitch() ScanArgs switch option exit routine ===*/

ScanArgsSwitch: procedure expose (Globals) curSw curSwArg

  select
  when curSw == 'd' then
    gDbgLvl = gDbgLvl + 1
  when curSw == 'e' then
    gChkEnv = 1
  when curSw == 'h' | curSw == '?' then
    call ScanArgsHelp
  when curSw == 't' then
    gTesting = 1
  when curSw == 'v' then
    gVerbose = 1
  when curSw == 'V' then do
    say gCmdName gVersion
    exit
  end
  otherwise
    call ScanArgsUsage 'switch '''curSw''' unexpected'
  end /* select */

  return

/* end ScanArgsSwitch */

/*=== ScanArgsArg() ScanArgs argument option exit routine ===*/

ScanArgsArg: procedure expose (Globals) curArg

  if gChkEnv then
    call ScanArgsUsage curArg 'not allowed with -e'

  if \ IsFile(curArg) then
    call ScanArgsUsage 'Cannot access' curArg 'file'

  i = gArgList.0 + 1
  gArgList.i = curArg
  gArgList.0 = i

  return

/* end ScanArgsArg */

/*=== ScanArgsTerm() ScanArgs scan end exit routine ===*/

ScanArgsTerm: procedure expose (Globals)

  if gArgList.0 = 0 & \ gChkEnv then do
    gArgList.1 = SysBootDrive() || '\CONFIG.SYS'
    gArgList.0 = 1
  end
  return

/* end ScanArgsTerm */

/*=== ScanArgsHelp() Display ScanArgs usage help exit routine ===*/

ScanArgsHelp:
  say
  say 'Check config.sys or environment.'
  say
  say 'Usage:' gCmdName '[-d] [-e] [-h] [-v] [-V] [-?] [configsysfile]'
  say
  say '  -d           Enable debug logic, repeat for more verbosity'
  say '  -e           Check environment, not config.sys file'
  say '  -h -?        Display this message'
  say '  -v           Enable verbose output'
  say '  -V           Display version number and quit'
  say
  say '  cfgsysfile   File to check, defaults to config.sys on boot drive'
  exit 255

/* end ScanArgsHelp */

/*=== ScanArgsUsage(message) Report Scanargs usage error exit routine ===*/

ScanArgsUsage:
  parse arg msg
  say
  if msg \== '' then
    say msg
  say 'Usage:' gCmdName '[-d] [-e] [-h] [-v] [-V] [-?] [configsysfile]'
  exit 255

/* end ScanArgsUsage */

/*==============================================================================*/
/*=== SkelRexxFunc standards - Delete unused - Move modified above this mark ===*/
/*==============================================================================*/

/*=== CatchError() Catch condition; return gErrCondition ===*/

CatchError:
  gErrCondition = condition('C')
  return
/* end CatchError */

/*=== IsDir(dirName[, full]) return true if directory is valid, accessible directory ===*/

IsDir: procedure
  /* If arg(2) not omitted, return full directory name or empty string */
  parse arg dir, full
  newdir = ''

  do 1
    if dir == '' then do
      cwd = ''				/* No restore needed */
      leave
    end
    dir = translate(dir, '\', '/')	/* Convert to OS/2 slashes */
    s = strip(dir, 'T', '\')		/* Chop trailing slashes unless root */
    if s \== '' & right(s, 1) \== ":" then
      dir = s				/* Chop */
    drv = filespec('D', dir)
    cwd = directory()			/* Remember */
    /* If have drive letter and requested directory on some other drive */
    if drv \== '' & translate(drv) \== translate(left(cwd, 2)) then do
      /* Avoid slow failures and unwanted directory changes */
      drvs = SysDriveMap('A:')
      if pos(translate(drv), drvs) = 0 then
	leave				/* Unknown drive */
      if SysDriveInfo(drv) == '' then
	leave				/* Drive not ready */
      cwd2 = directory(drv)		/* Remember current directory on other drive */
      newdir = directory(dir)		/* Try to change and get full path name */
      call directory cwd2		/* Restore current directory on other drive */
      leave
    end

    /* If no drive letter or same drive and not UNC name */
    if left(dir, 2) \== '\\' then do
      newdir = directory(dir)		/* Try to change and get full path name */
      leave
    end

    /* UNC name - hopefully server is accessible or this will be slow
       Accept
	 \\server
	 \\server\
	 \\server\dir\
	 \\server\dir
     */
    cwd = ''				/* No restore needed */
    wc = dir
    if right(wc, 1) \== '\' then
      wc = wc || '\'
    i = lastpos('\', wc)
    if substr(wc, 3, 1) == '\' then
      leave				/* Malformed UNC - no server name */
    if pos('*', wc) > 0 | pos('?', wc) > 0 then
      leave				/* No wildcards allowed */
    call SysFileTree wc, 'files', 'O'
    if files.0 > 0 then do
      s = files.1			/* Exists and is not empty */
      i = lastpos('\', s)
      newdir = left(s, i - 1)		/* Extract directory name from full path name */
      leave
    end
    /* Try wildcarded directory name */
    wc = strip(wc, 'T', '\')
    i = lastpos('\', wc)
    base = substr(wc, i + 1)
    if base == '' then
      leave				/* Should have matched above */
    wc = substr(wc, 1, i) || '*' || base || '*'
    call SysFileTree wc, 'files', 'DO'
    do fileNum = 1 to files.0
      /* Find directory name is list */
      s = files.fileNum
      i = lastpos('\', s)
      s2 = substr(s, i + 1)
      if translate(base) == translate(s2) then do
	newdir = left(s, i - 1)
	leave
      end
    end /* i */
  end /* 1 */

  if cwd \== '' then
    call directory cwd			/* Restore original directory and drive */

  if full \== '' then
    ret = newdir			/* Return full directory name or empty string */
  else
    ret = newdir \== ''			/* Return true if valid and accessible */
  return ret

/* end IsDir */

/*=== IsDrvRdy(driveSpec) return 1 if ready; accepts pathnames ===*/

IsDrvRdy: procedure
  drv = arg(1)
  if length(drv) < 2 | substr(drv, 2, 1) \== ':' then
    ready = 1				/* If no drive spec, assume ready */
  else do
    drv = translate(substr(drv, 1, 1))
    /* If not a drive letter, say not ready */
    if drv < 'A' | drv > 'Z' then
      ready = 0				/* If bad drive spec, not ready */
    else do
      info = SysDriveInfo(drv)
      /* If not blank and non-zero size (i.e. vfdisk), ready */
      ready = info \== '' & word(info, 3) \= 0
    end
  end
  return ready

/* end IsDrvRdy */

/*=== IsFile(file) return true if arg is file and file exists ===*/

IsFile: procedure expose (Globals)
  parse arg file
  if file == '' then
    yes = 0
  else do
    /* '.' and '..' returns files in '.' or '..' - so avoid false positives */
    call SysFileTree file, 'files', 'F'
    if RESULT \= 0 then
      call Die 'SysFileTree' file 'failed'
    /* Assume caller knows if arg contains wildcards */
    yes = file \== '.' & file \== '..' & files.0 \= 0
  end
  return yes

/* end IsFile */

/*==========================================================================*/
/*=== SkelRexx standards - Delete unused - Move modified above this mark ===*/
/*==========================================================================*/

/*=== Die([message,...]) Write multi-line message to STDERR and die ===*/

Die:
  call lineout 'STDERR', ''
  do i = 1 to arg()
    call lineout 'STDERR', arg(i)
  end
  call lineout 'STDERR', gCmdName 'aborting at line' SIGL || '.'
  call beep 200, 300
  call SysSleep 2
  exit 254

/* end Die */

/*=== Error() Set gErrCondition; report to STDOUT; trace and exit or return if called ===*/

Error:
  say
  parse source . . cmd
  gErrCondition = condition('C')
  say gErrCondition 'signaled at line' SIGL 'of' cmd || '.'
  if condition('D') \== '' then
    say 'REXX reason =' condition('D') || '.'
  if gErrCondition == 'SYNTAX' & symbol('RC') == 'VAR' then
    say 'REXX error =' RC '-' errortext(RC) || '.'
  else if symbol('RC') == 'VAR' then
    say 'RC =' RC || '.'
  say 'Source =' sourceline(SIGL)

  if condition('I') \== 'CALL' | gErrCondition == 'NOVALUE' | gErrCondition == 'SYNTAX' then do
    trace '?A'
    say 'Enter REXX commands to debug failure.  Press enter to exit script.'
    nop
    if symbol('RC') \== 'VAR' then
      RC = 255
    exit RC
  end

  return

/* end Error */

/*=== Halt() Report HALT condition to STDOUT and exit ===*/

Halt:
  say
  parse source . . cmd
  say condition('C') 'signaled at' cmd 'line' SIGL || '.'
  say 'Source =' sourceline(SIGL)
  say 'Sleeping for 2 seconds...'
  call SysSleep 2
  exit 253

/* end Halt */

/*=== LoadRexxUtil() Load RexxUtil functions ===*/

LoadRexxUtil:
  if RxFuncQuery('SysLoadFuncs') then do
    call RxFuncAdd 'SysLoadFuncs', 'REXXUTIL', 'SysLoadFuncs'
    if RESULT then
      call Die 'Cannot load SysLoadFuncs.'
    call SysLoadFuncs
  end
  return

/* end LoadRexxUtil */

/*=== ScanArgs(cmdLine) Scan command line ===*/

ScanArgs: procedure expose (Globals)

  /* Calls user exits to process arguments and switches */

  parse arg cmdTail
  cmdTail = strip(cmdTail)

  call ScanArgsInit

  /* Ensure optional settings initialized */
  if symbol('SWCTL') \== 'VAR' then
    swCtl = ''				/* Switches that take args, append ? if optional */
  if symbol('KEEPQUOTED') \== 'VAR' then
    keepQuoted = 0			/* Set to 1 to keep arguments quoted */

  /* Scan */
  curArg = ''				/* Current arg string */
  curSwList = ''			/* Current switch list */
  /* curSwArg = '' */			/* Current switch argument, if needed */
  noMoreSw = 0				/* End of switches */

  do while cmdTail \== '' | curArg \== '' | curSwList \== ''

    /* If arg buffer empty, refill */
    if curArg == '' then do
      qChar = left(cmdTail, 1)		/* Remember quote */
      if \ verify(qChar,'''"', 'M') then
	parse var cmdTail curArg cmdTail	/* Not quoted */
      else do
	/* Arg is quoted */
	curArg = ''
	do forever
	  /* Parse dropping quotes */
	  parse var cmdTail (qChar)quotedPart(qChar) cmdTail
	  curArg = curArg || quotedPart
	  /* Check for escaped quote within quoted string (i.e. "" or '') */
	  if left(cmdTail, 1) \== qChar then do
	    cmdTail = strip(cmdTail)	/* Strip leading whitespace */
	    leave			/* Done with this quoted arg */
	  end
	  curArg = curArg || qChar	/* Append quote */
	  if keepQuoted then
	    curArg = curArg || qChar	/* Append escaped quote */
	  parse var cmdTail (qChar) cmdTail	/* Strip quote */
	end /* do forever */
	if keepQuoted then
	  curArg = qChar || curArg || qChar	/* requote */
      end /* if quoted */
    end /* if curArg empty */

    /* If switch buffer empty, refill */
    if curSwList == '' & \ noMoreSw then do
      if left(curArg, 1) == '-' & curArg \== '-' then do
	if curArg == '--' then
	  noMoreSw = 1
	else
	  curSwList = substr(curArg, 2)	/* Remember switch string */
	curArg = ''			/* Mark empty */
	iterate				/* Refill arg buffer */
      end /* if switch */
    end /* if curSwList empty */

    /* If switch in progress */
    if curSwList \== '' then do
      curSw = left(curSwList, 1)	/* Next switch */
      curSwList = substr(curSwList, 2)	/* Drop from pending */
      /* Check switch allows argument, avoid matching ? */
      if pos(curSw, translate(swCtl,,'?')) \= 0 then do
	if curSwList \== '' then do
	  curSwArg = curSwList		/* Use rest of switch string for switch argument */
	  curSwList = ''
	end
	else if curArg \== '' & left(curArg, 1) \== '-' then do
	  curSwArg = curArg		/* Arg string is switch argument */
	  curArg = ''			/* Mark arg string empty */
	end
	else if pos(curSw'?', swCtl) = 0 then
	  call ScanArgsUsage 'Switch "-' || curSw || '" requires an argument'
	else
	  curSwArg = ''			/* Optional arg omitted */
      end

      call ScanArgsSwitch		/* Passing curSw and curSwArg */
      drop curSwArg			/* Must be used by now */
    end /* if switch */

    /* If arg */
    else if curArg \== '' then do
      noMoreSw = 1
      call ScanArgsArg			/* Passing curArg */
      curArg = ''
    end

  end /* while not done */

  call ScanArgsTerm

  return

/* end ScanArgs */

/*=== SetCmdName() Set gCmdName to short script name ===*/

SetCmdName: procedure expose (Globals)
  parse source . . cmd
  cmd = filespec('N', cmd)		/* Chop path */
  c = lastpos('.', cmd)
  if c > 1 then
    cmd = left(cmd, c - 1)		/* Chop extension */
  gCmdName = translate(cmd, xrange('a', 'z'), xrange('A', 'Z'))	/* Lowercase */
  return

/* end SetCmdName */

/* eof */
